import { Controller, UseGuards } from "@nestjs/common";
import { Crud, CrudController } from '@dataui/crud';
import { ZonaPesca } from "../entities/zona.pesca.entity";
import { ZonaPescaService } from "./zona_pesca.service";
import { ApiBody, ApiExtraModels, ApiOperation, ApiResponse, ApiTags, getSchemaPath } from '@nestjs/swagger';
import { ZonaPescaDTO, ZonaPescaUpdateDTO } from './zona_pesca.dto';
import { AuthGuard } from "../auth/auth.guard";
import { RolesGuard } from "../auth/roles.guard";
import { AdminAccess } from "../auth/decorators/admin.detorator";

@ApiExtraModels(ZonaPescaDTO, ZonaPescaUpdateDTO)
@ApiTags('Zona de Pesca')
@Crud({
  model: {
    type: ZonaPesca,
  },

  routes: {
    only: [
      'createOneBase',
      'getManyBase',
      'getOneBase',
      'updateOneBase',
      'deleteOneBase',
    ],
    deleteOneBase: {
      decorators: [
        AdminAccess(),
        ApiOperation({
          summary: 'Borra el registro zona de pesca seleccionado',
          description: 'Elimina un registro de zona de pesca.',
        }),
        ApiBody({
          description: 'Datos necesarios para borrar una zona de pesca',
          schema: { $ref: getSchemaPath(ZonaPescaDTO) },
        }),
        ApiResponse({
          status: 200,
          description: 'Entidad eliminada correctamente',
          schema: { $ref: getSchemaPath(ZonaPescaDTO) },
        }),
      ],
    },
    updateOneBase: {
      decorators: [
        AdminAccess(),
        ApiOperation({
          summary: 'Modifica el registro de zona de pesca seleccionado',
          description: 'Modifica el registro de zona de pesca.',
        }),
        ApiBody({
          description: 'Datos necesarios para modificar una zona de pesca',
          schema: { $ref: getSchemaPath(ZonaPescaUpdateDTO) },
        }),
        ApiResponse({
          status: 200,
          description: 'Entidad modificada correctamente',
          schema: { $ref: getSchemaPath(ZonaPescaUpdateDTO) },
        }),
      ],
    },
    createOneBase: {
      decorators: [
        AdminAccess(),
        ApiOperation({
          summary: 'Crea una nueva zona de pesca',
          description: 'Crea una zona de pesca.',
        }),
        ApiBody({
          description: 'Datos necesarios para crear una nueva zona de pesca',
          schema: { $ref: getSchemaPath(ZonaPescaDTO) },
        }),
        ApiResponse({
          status: 201,
          description: 'Entidad creada correctamente',
          schema: { $ref: getSchemaPath(ZonaPescaDTO) },
        }),
      ],
    },
    getOneBase: {
      decorators: [
        ApiOperation({
          summary: 'Busca la zona de pesca seleccionada',
          description: 'Buscar una zona de pesca.',
        }),
        ApiBody({
          description: 'Datos necesarios para buscar una zona de pesca',
          schema: { $ref: getSchemaPath(ZonaPescaDTO) },
        }),
        ApiResponse({
          status: 200,
          description: 'Entidad encontrada correctamente',
          schema: { $ref: getSchemaPath(ZonaPescaDTO) },
        }),
      ],
    },
    getManyBase: {
      decorators: [
        ApiOperation({
          summary: 'Obtener una lista de zonas de pesca',
          description: 'Retorna una lista de zonas de pesca.',
        }),
        ApiOperation({
          summary: 'Listar todas las zonas de pesca',
          description: 'Retorna una lista de zonas de pesca.',
        }),
        ApiResponse({
          status: 200,
          description: 'Lista de zonas de pesca encontrada correctamente.',
          schema: {
            type: 'array',
            items: { $ref: getSchemaPath(ZonaPescaDTO) },
          },
        }),
      ],
    },
  },
})

@UseGuards(AuthGuard, RolesGuard)
@Controller('zona-pesca')
export class ZonaPescaController implements CrudController<ZonaPesca> {
  constructor(public service: ZonaPescaService) {}
}
