import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import * as bcrypt from 'bcrypt';
import { DeleteResult, Entity, In, Repository, UpdateResult } from 'typeorm';
import { UsersEntity } from '../entities/users.entity';
import { UsersDTO, UsersUpdateDTO } from './users.dto';
import { ErrorManager } from 'src/config/error.manager';

@Injectable()
export class UsersService {
    constructor(
            @InjectRepository(UsersEntity) private readonly userRepository: Repository<UsersEntity>
    ){}

    public async createUser(body: UsersDTO): Promise<UsersEntity>{
        try{
            const newUser = await this.userRepository.create(body)
            const newPassword = await bcrypt.hash(newUser.password, +process.env.HASH_SALT)
            newUser.password = newPassword
            return await this.userRepository.save(newUser)

        } catch (error) {
            throw ErrorManager.createSignatureError(error.message)
        }
    }

    public async findUsers(): Promise<UsersEntity[]>{
        try{
            const users: UsersEntity[] = await this.userRepository.find()
            if(users.length === 0){
                throw new ErrorManager({
                    type:'BAD_REQUEST',
                    message: 'No se encontro resultado'
                })
            }
            return users

        } catch (error) {
            throw ErrorManager.createSignatureError(error.message)
        }
    }

    public async findUserById(id: string): Promise<UsersEntity>{
        try{
            const user : UsersEntity = await this.userRepository
            .createQueryBuilder('user')
            .where({id})
            .getOne();

            if(!user){
                throw new ErrorManager({
                    type: 'BAD_REQUEST',
                    message: 'No se encontro resultado',
                })
            }
            return user

        } catch (error) {
            throw ErrorManager.createSignatureError(error.message)
        }
    }

    public async findBy({key, value}:{
        key: keyof UsersDTO;
        value: any
    }){
        try {
            const user: UsersEntity = await this.userRepository
            .createQueryBuilder('user')
            .addSelect('user.password')
            .where({[key]:value})
            .getOne();
        return user;

        } catch (error){
            throw ErrorManager.createSignatureError(error.message)
        }
    }

    public async updateUser (body: UsersUpdateDTO, id: string): Promise<UpdateResult | undefined>{
        try{
            const user: UpdateResult = await this.userRepository.update(id, body)
            if(user.affected === 0){
                throw new ErrorManager({
                    type:'BAD_REQUEST',
                    message: 'No se pudo actualizar'
                })
            }
            return user;
        } catch (error) {
            throw ErrorManager.createSignatureError(error.message)
        }
    }

    public async deleteUser (id: string): Promise<DeleteResult | undefined>{
        try{
            const user: DeleteResult = await this.userRepository.delete(id)
            if(user.affected === 0){
                throw new ErrorManager({
                    type:'BAD_REQUEST',
                    message: 'No se pudo borrar'
                })
            }
            return user;
        } catch (error) {
            throw ErrorManager.createSignatureError(error.message)
        }
    }
}
