import { Controller, UseGuards } from "@nestjs/common";
import { Crud, CrudController } from '@dataui/crud';
import { Puerto } from '../entities/puerto.entity';
import { PuertoService } from './puerto.service';
import { ApiBody, ApiExtraModels, ApiOperation, ApiResponse, ApiTags, getSchemaPath } from "@nestjs/swagger";
import { PuertoDTO, PuertoUpdateDTO } from './puerto.dto';
import { AdminAccess } from "../auth/decorators/admin.detorator";
import { AuthGuard } from "../auth/auth.guard";
import { RolesGuard } from "../auth/roles.guard";

@ApiExtraModels(PuertoDTO, PuertoUpdateDTO)
@ApiTags('Puerto')
@Crud({
  model: {
    type: Puerto,
  },

  routes: {
    only: [
      'createOneBase',
      'getManyBase',
      'getOneBase',
      'updateOneBase',
      'deleteOneBase',
    ],
    deleteOneBase: {
      decorators: [
        AdminAccess(),
        ApiOperation({
          summary: 'Borra el registro del puerto seleccionado',
          description: 'Elimina un registro de puerto.',
        }),
        ApiBody({
          description: 'Datos necesarios para borrar un puerto',
          schema: { $ref: getSchemaPath(PuertoDTO) },
        }),
        ApiResponse({
          status: 200,
          description: 'Entidad eliminada correctamente',
          schema: { $ref: getSchemaPath(PuertoDTO) },
        }),
      ],
    },
    updateOneBase: {
      decorators: [
        AdminAccess(),
        ApiOperation({
          summary: 'Modifica el registro del puerto seleccionado',
          description: 'Modifica el registro de puerto.',
        }),
        ApiBody({
          description: 'Datos necesarios para modificar un registro de puerto',
          schema: { $ref: getSchemaPath(PuertoUpdateDTO) },
        }),
        ApiResponse({
          status: 200,
          description: 'Entidad modificada correctamente',
          schema: { $ref: getSchemaPath(PuertoUpdateDTO) },
        }),
      ],
    },
    createOneBase: {
      decorators: [
        AdminAccess(),
        ApiOperation({
          summary: 'Crea un nuevo puerto',
          description: 'Crea un puerto.',
        }),
        ApiBody({
          description: 'Datos necesarios para crear un registro de puerto',
          schema: { $ref: getSchemaPath(PuertoDTO) },
        }),
        ApiResponse({
          status: 201,
          description: 'Entidad creada correctamente',
          schema: { $ref: getSchemaPath(PuertoDTO) },
        }),
      ],
    },
    getOneBase: {
      decorators: [
        ApiOperation({
          summary: 'Busca el registro de puerto seleccionado',
          description: 'Buscar un puerto.',
        }),
        ApiBody({
          description: 'Datos necesarios para buscar un puerto',
          schema: { $ref: getSchemaPath(PuertoDTO) },
        }),
        ApiResponse({
          status: 200,
          description: 'Entidad encontrada correctamente',
          schema: { $ref: getSchemaPath(PuertoDTO) },
        }),
      ],
    },
    getManyBase: {
      decorators: [
        ApiOperation({
          summary: 'Obtener un listado de puertos',
          description: 'Retorna una lista de puertos.',
        }),
        ApiOperation({
          summary: 'Listar todos los puertos',
          description: 'Retorna una lista de puertos.',
        }),
        ApiResponse({
          status: 200,
          description: 'Lista de puertos encontrada correctamente.',
          schema: {
            type: 'array',
            items: { $ref: getSchemaPath(PuertoDTO) },
          },
        }),
      ],
    },
  },
})

@UseGuards(AuthGuard, RolesGuard)
@Controller('puerto')
export class PuertoController implements CrudController<Puerto> {
  constructor(public service: PuertoService) {}
}
