import { Controller, UseGuards } from "@nestjs/common";
import { Crud, CrudController } from '@dataui/crud';
import { Captura } from "../entities/captura.entity";
import { CapturaService } from "./captura.service";
import { ApiBody, ApiExtraModels, ApiOperation, ApiResponse, ApiTags, getSchemaPath } from '@nestjs/swagger';
import { CapturaDTO, CapturaUpdateDTO } from './captura.dto';
import { AdminAccess } from 'src/auth/decorators/admin.detorator';
import { AuthGuard } from "../auth/auth.guard";
import { RolesGuard } from "../auth/roles.guard";

@ApiExtraModels(CapturaDTO, CapturaUpdateDTO)
@ApiTags('Capturas')
@Crud({
  model: {
    type: Captura,
  },

  routes: {
    only: [
      'createOneBase',
      'getManyBase',
      'getOneBase',
      'updateOneBase',
      'deleteOneBase',
    ],
    deleteOneBase: {
      decorators: [
        AdminAccess(),
        ApiOperation({
          summary: 'Borra el registro seleccionado (solo usuarios ADMIN)',
          description: 'Elimina un registro de captura.',
        }),
        ApiBody({
          description: 'Datos necesarios para borrar una captura',
          schema: { $ref: getSchemaPath(CapturaDTO) },
        }),
        ApiResponse({
          status: 200,
          description: 'Entidad eliminada correctamente',
          schema: { $ref: getSchemaPath(CapturaDTO) },
        }),
      ],
    },
    updateOneBase: {
      decorators: [
        AdminAccess(),
        ApiOperation({
          summary: 'Modifica el registro de captura seleccionado',
          description: 'Modifica el registro de captura.',
        }),
        ApiBody({
          description: 'Datos necesarios para modificar un registro de captura',
          schema: { $ref: getSchemaPath(CapturaUpdateDTO) },
        }),
        ApiResponse({
          status: 200,
          description: 'Entidad modificada correctamente',
          schema: { $ref: getSchemaPath(CapturaUpdateDTO) },
        }),
      ],
    },
    createOneBase: {
      decorators: [
        AdminAccess(),
        ApiOperation({
          summary: 'Crea una nueva captura',
          description: 'Crea una captura.',
        }),
        ApiBody({
          description: 'Datos necesarios para crear un nuevo registro',
          schema: { $ref: getSchemaPath(CapturaDTO) },
        }),
        ApiResponse({
          status: 201,
          description: 'Entidad creada correctamente',
          schema: { $ref: getSchemaPath(CapturaDTO) },
        }),
      ],
    },
    getOneBase: {
      decorators: [
        ApiOperation({
          summary: 'Busca la captura seleccionada',
          description: 'Buscar una captura.',
        }),
        ApiBody({
          description: 'Datos necesarios para buscar una captura',
          schema: { $ref: getSchemaPath(CapturaDTO) },
        }),
        ApiResponse({
          status: 200,
          description: 'Entidad encontrada correctamente',
          schema: { $ref: getSchemaPath(CapturaDTO) },
        }),
      ],
    },
    getManyBase: {
      decorators: [
        ApiOperation({
          summary: 'Obtener una lista de capturas',
          description: 'Retorna una lista de capturas.',
        }),
        ApiOperation({
          summary: 'Listar todas las capturas',
          description: 'Retorna una lista de capturas.',
        }),
        ApiResponse({
          status: 200,
          description: 'Lista de capturas encontradas correctamente.',
          schema: {
            type: 'array',
            items: { $ref: getSchemaPath(CapturaDTO) },
          },
        }),
      ],
    },
  },
})

@UseGuards(AuthGuard, RolesGuard)
@Controller('captura')
export class CapturaController implements CrudController<Captura> {
  constructor(public service: CapturaService) {}
}
