import { CanActivate, ExecutionContext, Injectable, UnauthorizedException } from '@nestjs/common';
import { Reflector } from '@nestjs/core';
import { Request } from 'express';
import { PUBLIC_KEY } from 'src/constants/key-decorators';
import { UsersService } from 'src/users/users.service';
import { IUseToken } from './interfaces/auth.interface';
import { useToken } from 'src/config/use.token';

@Injectable()
export class AuthGuard implements CanActivate {
  constructor(
    private readonly UsersService: UsersService,
    private readonly reflector: Reflector,
  ) {}

  async canActivate(context: ExecutionContext): Promise<boolean> {
    const isPublic = this.reflector.get<boolean>(
      PUBLIC_KEY,
      context.getHandler(),
    );

    if (isPublic) {
      return true;
    }

    const req = context.switchToHttp().getRequest<Request>();

    // Obtener el encabezado 'Authorization'
    const authHeader = req.headers['authorization'];

    // Validar que el encabezado esté presente y siga el esquema 'Bearer'
    if (!authHeader || Array.isArray(authHeader)) {
      throw new UnauthorizedException('Authorization header is missing or invalid');
    }

    const tokenParts = authHeader.split(' ');
    if (tokenParts[0] !== 'Bearer' || tokenParts.length !== 2) {
      throw new UnauthorizedException('Invalid token format');
    }

    const token = tokenParts[1];

    // Procesar el token
    const manageToken: IUseToken | string = useToken(token);
    if (typeof manageToken === 'string') {
      throw new UnauthorizedException(manageToken);
    }

    if (manageToken.isExpired) {
      throw new UnauthorizedException('Token has expired');
    }

    const { sub } = manageToken;
    const user = await this.UsersService.findUserById(sub);
    if (!user) {
      throw new UnauthorizedException('User not found');
    }

    const userId = user.id.toString();
    req.idUser = userId; // Asigna el ID del usuario al request
    req.roleUser = user.role; // Asigna el rol del usuario al request

    return true;
  }
}
